/*
* Creation date : Wed Mar 07 12:49:07 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
* CE2_RSA_KG functions.
*
* \version CE2_RSA_KG.c#1:csrc:1
* \author Pavel Sasunkevich
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "CE2_RSA_KG.h"
#include "LLF_RSA_KG.h"
#include "tomcrypt.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  CE2_RSA_KG_GenerateKeyPair
*
* Inputs:
*  @param PubExp_ptr [in] - The pointer to the public exponent (public key);
*  @param PubExpSizeInBytes [in] - The public exponent size in bits;
*  @param KeySize [in] - The size of the key, in bits;
*  @param UserPrivKey_ptr [out] - A pointer to the private key structure;
*  @param UserPubKey_ptr  [out] - A pointer to the public key structure.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_RSA_INVALID_EXPONENT_POINTER_ERROR,
*     - CE2_RSA_INVALID_PRIV_KEY_STRUCT_POINTER_ERROR,
*     - CE2_RSA_INVALID_PUB_KEY_STRUCT_POINTER_ERROR,
*     - CE2_RSA_INVALID_MODULUS_SIZE,
*     - CE2_RSA_INVALID_EXPONENT_SIZE.
*
* \brief \b
* Description:
*  CE2_RSA_KG_GenerateKeyPair generates a Pair of public and
*  private keys on non CRT mode.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to generates a Pair of public and private keys on
*     non CRT mode key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_KG_GenerateKeyPair(
                        DxUint8_t            *PubExp_ptr,		/* in */
                        DxUint16_t            PubExpSizeInBytes,/* in */
                        DxUint32_t            KeySize,			/* in */
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,	/* out */
                        CE2_RSAUserPubKey_t  *UserPubKey_ptr)	/* out */
{
	if (PubExp_ptr == DX_NULL)
		return CE2_RSA_KG_GENKEYPAIR_INVALID_PUBEXP_PTR_ERROR;

	if (UserPrivKey_ptr == DX_NULL)
		return CE2_RSA_KG_GENKEYPAIR_INVALID_PRIVKEY_PTR_ERROR;

	if (UserPubKey_ptr == DX_NULL)
		return CE2_RSA_KG_GENKEYPAIR_INVALID_PUBKEY_PTR_ERROR;

	if (CE2_RSA_MIN_VALID_KEY_SIZE_VALUE_IN_BITS > KeySize ||
        KeySize > CE2_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS ||
        (KeySize % CE2_RSA_VALID_KEY_SIZE_MULTIPLE_VALUE_IN_BITS != 0))
		return CE2_RSA_KG_GENKEYPAIR_INVALID_MODULUS_SIZE_ERROR;

	if (PubExpSizeInBytes > sizeof(long))
		return CE2_RSA_KG_GENKEYPAIR_INVALID_EXPONENT_SIZE_ERROR;

	return LLF_RSA_KG_GenerateKeyPair(PubExp_ptr, PubExpSizeInBytes,
		KeySize, UserPrivKey_ptr, UserPubKey_ptr);
} /* End of CE2_RSA_KG_GenerateKeyPair */

/**
****************************************************************
* Function Name: 
*  CE2_RSA_KG_GenerateKeyPairCRT
*
* Inputs:
*  @param PubExp_ptr [in] - The pointer to the public exponent (public key);
*  @param PubExpSizeInBytes [in] - The public exponent size in bits;
*  @param KeySize [in] - The size of the key, in bits;
*  @param UserPrivKey_ptr [out] - A pointer to the private key structure;
*  @param UserPubKey_ptr  [out] - A pointer to the public key structure.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*     - CE2_RSA_INVALID_EXPONENT_POINTER_ERROR,
*     - CE2_RSA_INVALID_PRIV_KEY_STRUCT_POINTER_ERROR,
*     - CE2_RSA_INVALID_PUB_KEY_STRUCT_POINTER_ERROR,
*     - CE2_RSA_KEY_GEN_DATA_STRUCT_POINTER_INVALID,
*     - CE2_RSA_INVALID_MODULUS_SIZE,
*     - CE2_RSA_INVALID_EXPONENT_SIZE.
*
* \brief \b
* Description:
*  CE2_RSA_KG_GenerateKeyPairCRT generates a Pair of public and
*  private keys on CRT mode.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to build private key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_KG_GenerateKeyPairCRT(
                        DxUint8_t            *PubExp_ptr,		/* in */
                        DxUint16_t            PubExpSizeInBytes,/* in */
                        DxUint32_t            KeySize,			/* in */
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,	/* out */
                        CE2_RSAUserPubKey_t  *UserPubKey_ptr)	/* out */
{
	if (PubExp_ptr == DX_NULL)
		return CE2_RSA_KG_GENKEYPAIRCRT_INVALID_PUBEXP_PTR_ERROR;

	if (UserPrivKey_ptr == DX_NULL)
		return CE2_RSA_KG_GENKEYPAIRCRT_INVALID_PRIVKEY_PTR_ERROR;

	if (UserPubKey_ptr == DX_NULL)
		return CE2_RSA_KG_GENKEYPAIRCRT_INVALID_PUBKEY_PTR_ERROR;

	if (CE2_RSA_MIN_VALID_KEY_SIZE_VALUE_IN_BITS > KeySize ||
        KeySize > CE2_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS ||
        (KeySize % CE2_RSA_VALID_KEY_SIZE_MULTIPLE_VALUE_IN_BITS != 0))
		return CE2_RSA_KG_GENKEYPAIRCRT_INVALID_MODULUS_SIZE_ERROR;

	if (PubExpSizeInBytes > sizeof(long))
		return CE2_RSA_KG_GENKEYPAIRCRT_INVALID_EXPONENT_SIZE_ERROR;

	/* CE2_RSA_INVALID_MODULUS_SIZE */

	return LLF_RSA_KG_GenerateKeyPairCRT(PubExp_ptr, PubExpSizeInBytes,
		KeySize, UserPrivKey_ptr, UserPubKey_ptr);
} /* End of CE2_RSA_KG_GenerateKeyPairCRT */
